import type { NextApiRequest, NextApiResponse } from "next";
import { prisma } from "@/lib/prisma";

export default async function handler(
  req: NextApiRequest,
  res: NextApiResponse
) {
  if (req.method !== "GET") {
    return res.status(405).json({ ok: false, error: "Method not allowed" });
  }

  try {
    const { token } = req.query;

    if (!token || typeof token !== "string") {
      return res.status(400).json({
        ok: false,
        error: "Token is required",
        valid: false,
      });
    }

    // Find user by reset token
    const user = await prisma.user.findFirst({
      where: {
        passwordResetToken: token,
        passwordResetExpires: {
          gt: new Date(), // Token must not be expired
        },
      },
      select: { id: true, email: true },
    });

    if (!user) {
      return res.status(200).json({
        ok: true,
        valid: false,
        message: "Invalid or expired reset token",
      });
    }

    return res.status(200).json({
      ok: true,
      valid: true,
      email: user.email,
    });
  } catch (error: any) {
    console.error("Error verifying reset token:", error);
    return res.status(500).json({
      ok: false,
      error: "Failed to verify reset token",
      valid: false,
      details: error?.message,
    });
  }
}

